//----------------------------------------------------------------------------

#include <reg931.h>
#include "types.h"
#include "io.h"
#include "alt.h"
#include "uart.h"
#include "i2c.h"
#include "lcd.h"

//----------------------------------------------------------------------------

#define LCD_I2C_ADDR	(0x70 >> 1)

#define CMD_MODE_SET	0xCD
#define CMD_LOAD_DP	0x80
#define CMD_DEVICE_SEL	0xE0
#define CMD_BANK_SEL	0xF8
#define CMD_NOBLINK	0x70
#define CMD_BLINK	0x71

#define DASH            10
#define UNDERSCORE      11
#define SPACE           12
#define ALPHA_START     13

unsigned char const code LCD_Segs[] =
{
	0x6F, // 0x30 0
	0x03, // 0x31 1
	0x5D, // 0x32 2
	0x57, // 0x33 3
	0x33, // 0x34 4
	0x76, // 0x35 5
	0x7E, // 0x36 6
	0x43, // 0x37 7
	0x7F, // 0x38 8
	0x77, // 0x39 9
	0x10, //  -
	0x04, //  _
	0x00, //  space
	0x7B, // A
	0x3E, // b
	0x6C, // C
	0x1F, // d
	0x7C, // E
	0x78, // F
	0x77, // G
	0x3B, // H
	0x02, // i
	0x0F, // J
	0x38, // K - can't do
	0x2C, // L
	0x5A, // M - can't do
	0x1A, // n
	0x6F, // O
	0x79, // P
	0x75, // Q - can't do
	0x18, // r
	0x76, // S
	0x3C, // t
	0x0E, // u
	0x2F, // V - can't do
	0x4E, // W - can't do
	0x2B, // X - can't do
	0x37, // y
	0x5D, // Z

};

unsigned char idata LCD_Digits[NUM_LCD_DIGITS];
unsigned char LCD_Blink;

//----------------------------------------------------------------------------

void LCD_Update(void)
{
	unsigned char i,b;

	i = 0;
	I2C_Data[i++] = CMD_MODE_SET;
	I2C_Data[i++] = CMD_LOAD_DP;
	I2C_Data[i++] = CMD_DEVICE_SEL;
	I2C_Data[i++] = CMD_BANK_SEL;

	if(LCD_Blink) I2C_Data[i++] = CMD_BLINK;
	else          I2C_Data[i++] = CMD_NOBLINK;

	I2C_Data[i++] = (LCD_Digits[0] >> 4); // || LCD_BAR
	I2C_Data[i++] = (LCD_Digits[0] << 4) | (LCD_Digits[1] >> 3);
	I2C_Data[i++] = (LCD_Digits[1] << 5) | (LCD_Digits[2] >> 2);
	I2C_Data[i++] = (LCD_Digits[2] << 6) | (LCD_Digits[3] >> 1);
	I2C_Data[i++] = (LCD_Digits[3] << 7);

	I2C_NumBytes = i;  
	i2c_transmit(LCD_I2C_ADDR);

	do
	{
		b = i2c_getstatus();
	} while(b & I2C_BUSY);
}

//----------------------------------------------------------------------------

void LCD_Init(void)
{
	unsigned char b;

	LCD_Blink = 0;

	LCD_Digits[0] = LCD_Segs[SPACE];
	LCD_Digits[1] = LCD_Segs[SPACE];
	LCD_Digits[2] = LCD_Segs[SPACE];
	LCD_Digits[3] = LCD_Segs[SPACE];

	I2C_NumBytes = 10;
	I2C_Data[0] = CMD_MODE_SET;
	I2C_Data[1] = CMD_LOAD_DP;
	I2C_Data[2] = CMD_DEVICE_SEL;
	I2C_Data[3] = CMD_BANK_SEL;
	I2C_Data[4] = CMD_NOBLINK;
	I2C_Data[5] = 0x05;
	I2C_Data[6] = 0xD5;
	I2C_Data[7] = 0x9B;
	I2C_Data[8] = 0xFF;
	I2C_Data[9] = 0x00;

	i2c_transmit(LCD_I2C_ADDR);

	do
	{
		b = i2c_getstatus();
	} while(b & I2C_BUSY);
}

//----------------------------------------------------------------------------

void LCD_Command(unsigned char cmd)
{
	switch(cmd)
	{
		case LCD_BLINK_OFF :
			LCD_Blink = 0;
			LCD_Update();
			break;
			
		case LCD_BLINK_ON :
			LCD_Blink = 1;
			LCD_Update();
			break;

		case LCD_CLEAR :
			LCD_Digits[0] = LCD_Segs[SPACE];
			LCD_Digits[1] = LCD_Segs[SPACE];
			LCD_Digits[2] = LCD_Segs[SPACE];
			LCD_Digits[3] = LCD_Segs[SPACE];
			
			LCD_Update();
			break;

		default :
			break;
	}
}

//----------------------------------------------------------------------------

char ConvertChar(char c)
{
	if((c >= 'a') && (c <= 'z')) c = c - 'a' + ALPHA_START;
	else
		if((c >= 'A') && (c <= 'Z')) c = c - 'A' + ALPHA_START;
		else
			if((c >= '0') && (c <= '9')) c = c - '0';
			else
				if(c == '-') c = DASH;
				else
					if(c == '_') c = UNDERSCORE;
					else c = SPACE;
	return c;
}

//----------------------------------------------------------------------------

void LCD_DispChar(unsigned char index, char c)
{

	LCD_Digits[index] = LCD_Segs[ConvertChar(c)];
	LCD_Update();
}

//----------------------------------------------------------------------------

void LCD_DispStr(char *s)
{
	unsigned char i,c;

	for(i=0;i<NUM_LCD_DIGITS;i++) LCD_Digits[i] = LCD_Segs[SPACE];

	i = 0;

	while((i < 4) && s[i])
	{
		c = LCD_Segs[ConvertChar(s[i])];
	
		LCD_Digits[i] = c;
		i++;
	}

	LCD_Update();
}

//----------------------------------------------------------------------------
